// -*-C++-*- time-segmented.cpp
// ----------------------------------------------------------------------- 
//  Copyright  2005 Dietmar Khl, All Rights Reserved                     
//                                                                         
//  Permission to use, copy, modify, distribute and sell this              
//  software for any purpose is hereby granted without fee, provided       
//  that the above copyright notice appears in all copies and that         
//  both that copyright notice and this permission notice appear in        
//  supporting documentation. Dietmar Khl makes no representations about  
//  the suitability of this software for any purpose. It is provided       
//  "as is" without express or implied warranty.                           
// ----------------------------------------------------------------------- 

// Author:  Dietmar Kuehl http://www.dietmar-kuehl.de 
// Title:   Timing tests between normal and segmented algorithms

// ----------------------------------------------------------------------- 

#include "stream.hpp"
#include "deque.hpp"
#include "algo.hpp"
#include "default_pm.hpp"
#include "timer.hpp"

#include <algorithm>
#include <iostream>
#include <iterator>
#include <streambuf>
#include <string>
#include <utility>
#include <vector>

#include <assert.h>
#include <stdlib.h>
#include <string.h>

// -----------------------------------------------------------------------

#if 1
long       total = 200000 * 1000;
#else
long       total = 200000 * 100;
#endif
long const size =  1000000;
int        loop = total / size;

// -----------------------------------------------------------------------
// Just repeat the test a number of times and return the accumulated
// time.

template <typename F>
double time_test(F const& f)
{
  cool::timer t;
  for (int i = 0; i < loop; ++i)
    f.run();
  return t.elapsed();
}

// -----------------------------------------------------------------------

template <typename Cont, typename It>
struct std_find_t {
  std_find_t(Cont& cont, It it): m_cont(cont), m_it(it) {}
  void run() const {
    std::find(m_cont.begin(), m_cont.end(), *m_it++);
  }
  Cont&      m_cont;
  mutable It m_it;
};

template <typename Cont, typename It>
std_find_t<Cont, It> std_find(Cont& cont, It it)
{
  return std_find_t<Cont, It>(cont, it);
}

template <typename Cont, typename It>
struct segment_find_t {
  segment_find_t(Cont& cont, It it): m_cont(cont), m_it(it) {}
  void run() const {
    cool::find(m_cont.begin(), m_cont.end(), cool::default_pm(), *m_it++);
  }
  Cont&      m_cont;
  mutable It m_it;
};

template <typename Cont, typename It>
segment_find_t<Cont, It> segment_find(Cont& cont, It it)
{
  return segment_find_t<Cont, It>(cont, it);
}

template <typename Cont, typename It>
void
find_test(Cont& cont, It it)
{
  std::cout << "  " << cool::fmt(6, 3) << time_test(std_find(cont, it));
  std::cout << "  " << cool::fmt(6, 3) << time_test(segment_find(cont, it));
  std::cout << "\n";
}

// -----------------------------------------------------------------------

template <typename Cont, typename Func>
struct std_for_each_t {
  std_for_each_t(Cont& cont, Func func): m_cont(cont), m_func(func) {}
  void run() const {
    std::for_each(m_cont.begin(), m_cont.end(), m_func);
  }
  Cont& m_cont;
  Func  m_func;
};

template <typename Cont, typename Func>
std_for_each_t<Cont, Func> std_for_each(Cont& c, Func f)
{
  return std_for_each_t<Cont, Func>(c, f);
}

template <typename Cont, typename Func>
struct segment_for_each_t {
  segment_for_each_t(Cont& cont, Func func): m_cont(cont), m_func(func) {}
  void run() const {
    cool::for_each(m_cont.begin(), m_cont.end(), cool::default_pm(), m_func);
  }
  Cont& m_cont;
  Func  m_func;
};

template <typename Cont, typename Func>
segment_for_each_t<Cont, Func> segment_for_each(Cont& c, Func f)
{
  return segment_for_each_t<Cont, Func>(c, f);
}

struct nop { void operator()(int) {} };

int s_count = 0;
struct count { void operator()(int) { ++s_count; } };

unsigned long s_sum = 0;
struct sum { void operator()(int a) { s_sum += a; } };


template <typename Cont>
void
for_each_test(Cont& cont)
{
  std::cout << "  " << cool::fmt(6,3) << time_test(std_for_each(cont, nop()));
  std::cout << "  " << cool::fmt(6,3) << time_test(segment_for_each(cont, nop()));
  std::cout << "   ";
  std::cout << "  " << cool::fmt(6,3) << time_test(std_for_each(cont, count()));
  std::cout << "  " << cool::fmt(6,3) << time_test(segment_for_each(cont, count()));
  std::cout << "   ";
  std::cout << "  " << cool::fmt(6,3) << time_test(std_for_each(cont, sum()));
  std::cout << "  " << cool::fmt(6,3) << time_test(segment_for_each(cont, sum()));
  std::cout << "\n";
}

// -----------------------------------------------------------------------

template <typename C1, typename C2>
struct std_copy_t
{
  std_copy_t(C1& c1, C2& c2): m_c1(c1), m_c2(c2) {}
  void run() const {
    std::copy(m_c1.begin(), m_c1.end(), m_c2.begin());
  }
  C1& m_c1;
  C2& m_c2;
};

template <typename C1, typename C2>
std_copy_t<C1, C2> std_copy(C1& c1, C2& c2)
{
  return std_copy_t<C1, C2>(c1, c2);
}

template <typename C1, typename C2>
struct segment_copy_t
{
  segment_copy_t(C1& c1, C2& c2): m_c1(c1), m_c2(c2) {}
  void run() const {
    cool::copy(m_c1.begin(), m_c1.end(), cool::default_pm(), m_c2.begin(), cool::default_pm());
  }
  C1& m_c1;
  C2& m_c2;
};

template <typename C1, typename C2>
segment_copy_t<C1, C2> segment_copy(C1& c1, C2& c2)
{
  return segment_copy_t<C1, C2>(c1, c2);
}

template <typename C1, typename C2>
void
copy_test(C1& c1, C2& c2)
{
  std::cout << "  " << cool::fmt(6,3) << time_test(std_copy(c1, c2));
  std::cout << "  " << cool::fmt(6,3) << time_test(segment_copy(c1, c2));
  std::cout << "\n";
}

// -----------------------------------------------------------------------

template <typename T, int sz> T const* begin(T(&a)[sz]) { return a; }
template <typename T, int sz> T const* end(T(&a)[sz]) { return a + sz-1; }

// -----------------------------------------------------------------------

int main()
{
  char a[] = "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ";
  std::vector<char> init(size);
  std::vector<char> vc(size);
  std::vector<char>::iterator it = init.begin();
  while (end(a) - begin(a) < init.end() - it)
    it = std::copy(begin(a), end(a), it);
  std::copy(begin(a), begin(a) + (init.end() - it), it);

  cool::stream<true> stream1(init, 512);
  cool::stream<false> stream2(init, 512);
  std::deque<char> dc(init.begin(), init.end());
  std::deque<char> dc2(init.begin(), init.end());

  std::vector<int> vi;
  for (int i = 0; i < size; ++i)
    vi.push_back(std::rand());
  std::deque<int>  di(vi.begin(), vi.end());

  std::cout << std::unitbuf;

  assert(loop < size);

  std::cout << "find():         std     seg\n";
  std::cout << "vector<int>  "; find_test(vi, vi.rbegin());
  std::cout << "deque<int>   "; find_test(di, vi.rbegin());
  std::cout << "\n";

  std::cout << "copy():                           std     seg\n";
  std::cout << "vector<char>/vector<char>:     "; copy_test(init, vc);
  std::cout << "vector<char>/deque<char>:      "; copy_test(vc, dc);
  std::cout << "deque<char>/vector<char>:      "; copy_test(dc, vc);
  std::cout << "deque<char>/deque<char>:       "; copy_test(dc, dc2);
  std::cout << "istreambuf/vector<char>:       "; copy_test(stream1, vc);
  std::cout << "istreambuf/deque<char>:        "; copy_test(stream1, dc);
  std::cout << "seg istreambuf/vector<char>:   "; copy_test(stream2, vc);
  std::cout << "seg istreambuf/deque<char>:    "; copy_test(stream2, dc);
  std::cout << "\n";

  std::cout << "for_each():          "
	    << "std/nop seg/nop    "
	    << "std/cou seg/cou    "
	    << "std/sum seg/sum    "
	    << "\n";
  std::cout << "std::vector        "; for_each_test(init);
  std::cout << "std::deque<int>    "; for_each_test(di);
  std::cout << "std::deque<char>   "; for_each_test(dc);
  std::cout << "std::streambuf     "; for_each_test(stream1);
  std::cout << "segment::streambuf "; for_each_test(stream2);
  std::cout << "\n";
}
