// -*-C++-*- stream.hpp
// ----------------------------------------------------------------------- 
//  Copyright  2005 Dietmar Khl, All Rights Reserved                     
//                                                                         
//  Permission to use, copy, modify, distribute and sell this              
//  software for any purpose is hereby granted without fee, provided       
//  that the above copyright notice appears in all copies and that         
//  both that copyright notice and this permission notice appear in        
//  supporting documentation. Dietmar Khl makes no representations about  
//  the suitability of this software for any purpose. It is provided       
//  "as is" without express or implied warranty.                           
// ----------------------------------------------------------------------- 

// Author:  Dietmar Kuehl http://www.dietmar-kuehl.de 
// Title:   

// ----------------------------------------------------------------------- 

#if !defined(COOL_STREAM_HPP)
#define COOL_STREAM_HPP 1

#include "segments.hpp"

#include <streambuf>
#include <vector>

namespace cool
{
  // ---------------------------------------------------------------------

  struct streambuf:
    std::streambuf
  {
    template <typename Cont>
    streambuf(Cont const& cont, int seg):
      m_buffer(cont.begin(), cont.end()),
      m_seg(seg),
      m_it(m_buffer.begin())
    {
      reset();
    }

    // access to protected streambuf data:
    char const* begin() const { return gptr(); }
    char const* end() const { return egptr(); }
    void set_gptr(char const* p) { gbump(p - gptr()); }
    bool next()
    {
      gbump(egptr() - gptr());
      return !traits_type::eq_int_type(traits_type::eof(), underflow());
    }
    void reset() {
      m_it = m_buffer.begin();
      setg(&*m_it, &*m_it,
	   &*m_it + (m_seg < std::streamsize(m_buffer.size())? m_seg
		     : m_buffer.size()));
    }

  private:
    int underflow()
    {
      if (m_seg < std::streamsize(m_buffer.end() - m_it))
	{
	  m_it += m_seg;
	  setg(&*m_it, &*m_it,
	       &*(m_seg < std::streamsize(m_buffer.end() - m_it)
		  ? m_it + m_seg : m_buffer.end()));
	}
      else
	{
	  m_it = m_buffer.end();
	  setg(&*m_it, &*m_it, &*m_it);
	}
      return gptr() != egptr()? traits_type::to_int_type(*gptr())
	: traits_type::eof();
    }

    std::vector<char>           m_buffer;
    std::streamsize             m_seg;
    std::vector<char>::iterator m_it;
  };

  // ---------------------------------------------------------------------

  struct istreambuf_iterator
  {
    typedef char                    value_type;
    typedef char const*             pointer;
    typedef char const&             reference;
    typedef std::streamsize         difference_type;
    typedef std::input_iterator_tag iterator_category;

    typedef std::char_traits<char> traits;
    istreambuf_iterator(): m_sbuf(), m_char(traits::eof()) {}
    istreambuf_iterator(streambuf* sb):
      m_sbuf(sb),
      m_char(sb->sgetc())
    {
      if (traits::eq_int_type(m_char, traits::eof()))
	m_sbuf = 0;
    }

    char operator*() const { return traits::to_char_type(m_char); }
    istreambuf_iterator& operator++() {
      m_char = m_sbuf->snextc();
      if (traits::eq_int_type(m_char, traits::eof()))
	m_sbuf = 0;
      return *this;
    }
    istreambuf_iterator operator++(int) {
      istreambuf_iterator rc(*this);
      ++*this;
      return rc;
    }

    bool operator== (istreambuf_iterator const& it) const {
      return m_sbuf == it.m_sbuf;
    }
    bool operator!= (istreambuf_iterator const& it) const {
      return m_sbuf != it.m_sbuf;
    }

    streambuf* m_sbuf;
    int        m_char;
  };

  // ---------------------------------------------------------------------

  template <>
  struct is_segmented<istreambuf_iterator>
  {
    enum { value = true };
  };

  struct istreambuf_segment_cursor
  {
    istreambuf_segment_cursor(cool::istreambuf_iterator const& it)
      {
        m_sbuf = it.m_sbuf;
      }

    bool operator== (istreambuf_segment_cursor const& it) const
      {
        return m_sbuf == it.m_sbuf;
      }
    bool operator!= (istreambuf_segment_cursor const& it) const
      {
        return !(*this == it);
      }

    istreambuf_segment_cursor& operator++()
      {
        if (!m_sbuf->next())
          m_sbuf = 0;
        return *this;
      }

    cool::streambuf* m_sbuf;
  };

  template <>
  struct segment_cursor<istreambuf_iterator>
  {
    typedef istreambuf_segment_cursor type;
  };

  template <>
  struct local_cursor<istreambuf_iterator>
  {
    typedef char const* type;
  };

  istreambuf_segment_cursor segment(cool::istreambuf_iterator const& it)
  {
    return istreambuf_segment_cursor(it);
  }

  char const* local(cool::istreambuf_iterator const& it)
  {
    return it.m_sbuf == 0? 0: it.m_sbuf->begin();
  }

  char const* local_begin(cool::istreambuf_segment_cursor const& it)
  {
    return it.m_sbuf == 0? 0: it.m_sbuf->begin();
  }

  char const* local_end(cool::istreambuf_segment_cursor const& it)
  {
    return it.m_sbuf == 0? 0: it.m_sbuf->end();
  }

  cool::istreambuf_iterator
  make_segmented(cool::istreambuf_segment_cursor const& s, char const* l)
  {
    s.m_sbuf->set_gptr(l);
    return cool::istreambuf_iterator(s.m_sbuf);
  }

  // ---------------------------------------------------------------------

  template <bool>
  struct config {
    typedef std::istreambuf_iterator<char> iterator;
  };

  template <>
  struct config<false> {
    typedef cool::istreambuf_iterator iterator;
  };

  // ---------------------------------------------------------------------

  template <bool b>
  struct stream
  {
    template <typename Cont>
    stream(Cont const& cont, int seg): m_sbuf(cont, seg) {}

    typedef typename config<b>::iterator iterator;

    iterator begin() { m_sbuf.reset(); return iterator(&m_sbuf); }
    iterator end() { return iterator(); }

    streambuf m_sbuf;
  };

  // ---------------------------------------------------------------------
} // namespace cool

#endif /* COOL_STREAM_HPP */
