// -*-C++-*- include/cool/algorithm/lexicographical_compare.hpp
// ----------------------------------------------------------------------- 
//  Copyright  2002 Dietmar Khl, All Rights Reserved                     
//                                                                         
//  Permission to use, copy, modify, distribute and sell this              
//  software for any purpose is hereby granted without fee, provided       
//  that the above copyright notice appears in all copies and that         
//  both that copyright notice and this permission notice appear in        
//  supporting documentation. Dietmar Khl makes no representations about  
//  the suitability of this software for any purpose. It is provided       
//  "as is" without express or implied warranty.                           
// ----------------------------------------------------------------------- 

// Author:  Dietmar Kuehl http://www.dietmar-kuehl.de 
// Title:   the lexicographical_compare() family of algorithms
// Version: $Id: lexicographical_compare.hpp,v 1.1.1.1 2003/01/07 00:15:08 kuehl Exp $ 

// ----------------------------------------------------------------------- 

#if !defined(COOL_ALGORITHM_LEXICOGRAPHICAL_COMPARE_HPP)
#define COOL_ALGORITHM_LEXICOGRAPHICAL_COMPARE_HPP 1

#if !defined(COOL_ALGORITHM_MISMATCH_HPP)
#  include "cool/algorithm/mismatch.hpp"
#endif
#if !defined(COOL_PROPERTY_IDENTITY_MAP_HPP)
#  include "cool/property/identity_map.hpp"
#endif
#if !defined(COOL_TRAITS_ITERATOR_HPP)
#  include "cool/traits/itertor.hpp"
#endif
#if !defined(COOL_TRAITS_PROPERTY_HPP)
#  include "cool/traits/property.hpp"
#endif
#if !defined(COOL_FUNCTIONAL_LESS_HPP)
#  include "cool/functional/less.hpp"
#endif

namespace cool
{

  // ------------------------------------------------------------------------

  template <typename BinaryPredicate>
  struct lexicographical_compare_pred_t
  {
    typedef bool result_type;
    typedef typename BinaryPredicate::first_argument_type first_argument_type;
    typedef typename BinaryPredicate::second_argument_type second_argument_type;
    
    lexicographical_compare_pred_t(BinaryPredicate pred): m_pred(pred) {}
    bool operator()(first_argument_type const& a1, second_argument_type const& a2) {
      return !m_pred(a1, a2) && !m_pred(a2, a1);
    }

    BinaryPredicate m_pred;
  };

  template <typename BinaryPredicate>
  inline cool::lexicographical_compare_pred_t<BinaryPredicate>
  lexicographical_compare_pred(BinaryPredicate pred)
  {
    return cool::lexicographical_compare_pred_t<BinaryPredicate>(pred);
  }

  // ------------------------------------------------------------------------

  template <typename InputIterator1, typename InputIterator2, typename BinaryPredicate,
	    typename ReadPM1, typename ReadPM2>
  inline bool
  lexicographical_compare(InputIterator1 begin1, InputIterator1 end1,
			  InputIterator2 begin2, InputIterator2 end2,
			  BinaryPredicate pred,
			  ReadPM1 pm1, ReadPM2 pm2)
  {
    std::pair<InputIterator1, InputIterator2> const& p = cool::mismatch(begin1, end1, begin2, end2,
									cool::lexicographical_compare_pred(pred), pm1, pm2);
    return p.first == end1
      ? p.second != end2
      : (p.second != end2 && pred(get(pm1, *p.first), get(pm2, *p.second)));
  }

  // ------------------------------------------------------------------------

  template <typename InputIterator1, typename InputIterator2, typename ReadPM1, typename ReadPM2>
  inline bool
  lexicographical_compare(InputIterator1 begin1, InputIterator1 end1,
			  InputIterator2 begin2, InputIterator2 end2,
			  ReadPM1 pm1, ReadPM2 pm2)
  {
    typedef typename cool::iterator_traits<InputIterator1>::key_type          key_type;
    typedef typename cool::property_use_traits<ReadPM1, key_type>::value_type value_type;

    return cool::lexicographical_compare(begin1, end1, begin2, end2, cool::less<value_type>(), pm1, pm2);
  }

  // ------------------------------------------------------------------------

  template <typename InputIterator1, typename InputIterator2, typename BinaryPredicate>
  inline bool
  lexicographical_compare(InputIterator1 begin1, InputIterator1 end1,
			  InputIterator2 begin2, InputIterator2 end2,
			  BinaryPredicate pred)
  {
    return cool::lexicographical_compare(begin1, end1, begin2, end2, pred, cool::identity_map(), cool::identity_map());
  }

  // ------------------------------------------------------------------------

  template <typename InputIterator1, typename InputIterator2>
  inline bool
  lexicographical_compare(InputIterator1 begin1, InputIterator1 end1,
			  InputIterator2 begin2, InputIterator2 end2)
  {
    return cool::lexicographical_compare(begin1, end1, begin2, end2, cool::identity_map(), cool::identity_map());
  }

  // ------------------------------------------------------------------------

} // namespace cool

#endif /* COOL_ALGORITHM_LEXICOGRAPHICAL_COMPARE_HPP */
