#ifndef algebraic_functions_include
#define algebraic_functions_include

#include <vector>
#include <algorithm>
#include <numeric>

#include "enable_if.hpp"
#include "property_traits.hpp"

// Pure algebraic functions (a little useless)

// {T, Op} must be a Magma
// T must be EqualityComparable
template <class T, class Op>
inline bool equalResults(const T& v1a, const T& v1b, 
			 const T& v2a, const T& v2b, Op op) {
  return op(v1a, v1b) == op(v2a, v2b);
}

// {T, Op} must be a Monoid
// T must be EqualityComparable
template <class T, class Op>
inline bool identityPair(const T& v1, const T& v2, Op op) {
  return op(v1, v2) == op.identity();
}

// {T, Op} must be a Monoid
template <class T, class Op>
inline T multiplyAndSquare(T base, int exp, Op op) {
  T value(op.identity()), square(base);
  for (; exp > 0; exp>>= 1) {
    if (exp & 1) value= op(value, square);
    square= op(square, square); }
  return value;  
} 

// {T, Op} must be a Group
// T must be LessThanComparable and Assignable
template <class T, class Op>
inline int poorMensDivision(const T& v1, const T& v2, Op op) {
  // copies to avoid redundant operations
  T id(op.identity()), iv2(op.inverse(v2)), tmp(v1);

  if (v1 <= op.identity()) return 0;
  int counter= 0;
  for (; tmp > id; counter++) tmp= op(tmp, iv2);
  if (tmp < id) counter--;
  return counter;
}

// {T, Op} must be a Group
// T must be LessThanComparable and Assignable
template <class T, class Op>
inline int poorMensAbsDivision(const T& v1, const T& v2, Op op) {
  // copies to avoid redundant operations
  T id(op.identity()), iv2(v2 < op.identity() ? v2 : op.inverse(v2)),
    va1(v1 < op.identity() ? op.inverse(v1) : v1), tmp(va1);
  if (va1 <= op.identity()) return 0;
  int counter= 0;
  for (; tmp > id; counter++) tmp= op(tmp, iv2);
  if (tmp < id) counter--;
  return counter;
}

// {Iter*, Op} must be a CommutativeMonoid
struct sortedAccumulate_t {
  template <class Iter, class Op, class Comp>
  typename enable_if<glas::is_op_associative<Op>::value && glas::is_op_commutative<Op>::value, 
		     typename std::iterator_traits<Iter>::value_type>::type
  operator() (Iter first, Iter last, Op op, Comp comp) {
    // cout << "sortedAccumulate_t\n";
    typedef typename std::iterator_traits<Iter>::value_type value_type;
    std::vector<value_type> tmp(first, last);
    std::sort(tmp.begin(), tmp.end(), comp); 
    return std::accumulate(tmp.begin(), tmp.end(), op.identity(), op); }
} sortedAccumulate;

// {Iter*, Op} must be a Monoid
struct unsortedAccumulate_t {
  template <class Iter, class Op>
  typename std::iterator_traits<Iter>::value_type
  operator() (Iter first, Iter last, Op op) {
    // cout << "unsortedAccumulate_t\n";
    return std::accumulate(first, last, op.identity(), op); }

  // Only for Compability
  template <class Iter, class Op, class Comp>
  typename std::iterator_traits<Iter>::value_type
  operator() (Iter first, Iter last, Op op, Comp) {
    return operator() (first, last, op); }
} unsortedAccumulate;
    
// {Iter*, Op} must be a Monoid
template <class Iter, class Op, class Comp>
inline typename std::iterator_traits<Iter>::value_type
trySortedAccumulate(Iter first, Iter last, Op op, Comp comp) {
  typename if_type<glas::is_op_associative<Op>::value && glas::is_op_commutative<Op>::value,  
                   sortedAccumulate_t, unsortedAccumulate_t>::type  accumulate;
  return accumulate(first, last, op, comp);
}

#endif // algebraic_functions_include
