// -*-C++-*- segmented.cpp
// ----------------------------------------------------------------------- 
//  Copyright © 2005 Dietmar Kühl, All Rights Reserved                     
//                                                                         
//  Permission to use, copy, modify, distribute and sell this              
//  software for any purpose is hereby granted without fee, provided       
//  that the above copyright notice appears in all copies and that         
//  both that copyright notice and this permission notice appear in        
//  supporting documentation. Dietmar Kühl makes no representations about  
//  the suitability of this software for any purpose. It is provided       
//  "as is" without express or implied warranty.                           
// ----------------------------------------------------------------------- 

// Author:  Dietmar Kuehl http://www.dietmar-kuehl.de 
// Title:   a file running some algorithms as test

// ----------------------------------------------------------------------- 

#include "algo.hpp"
#include "default_pm.hpp"
#include "array_traits.hpp"
#include <functional>
#include <iostream>
#include <iterator>
#include <vector>

// ----------------------------------------------------------------------- 

namespace multipass
{
  template <typename T>
  struct segment_cursor
  {
    segment_cursor(int no, T* start, T* end): m_no(no), m_start(start), m_end(end) {}

    int no() const { return m_no; }
    T*  begin() const { return m_start; }
    T*  end() const { return m_end; }

    segment_cursor& operator++() { --m_no; }
    bool operator== (segment_cursor const& c) const { return m_no == c.m_no; }
    bool operator!= (segment_cursor const& c) const { return !(*this == c); }

    int m_no;
    T*  m_start;
    T*  m_end;
  };

  template <typename T>
  struct cursor
  {
    cursor(segment_cursor<T> const& sc, T* lc):
      m_cursor(sc),
      m_current(lc)
    {
      if (m_current == local_end(m_cursor) && m_cursor.no() > 0)
      {
        ++m_cursor;
        m_current = local_begin(m_cursor);
      }
    }
    cursor(T* end):
      m_cursor(0, end, end),
      m_current(end)
    {
    }
    cursor(int no, T* start, T* end):
      m_cursor(no - 1, start, end),
      m_current(start)
    {
    }
    cursor(int no, T* start, T* end, T* current):
      m_cursor(no - 1, start, end),
      m_current(current)
    {
    }

    segment_cursor<T> segment() const { return m_cursor; }
    T*                local() const { return m_current; }

    cursor& operator++()
    {
      if (++m_current == local_end(m_cursor) && m_cursor.no() > 0)
      {
        ++m_cursor;
        m_current = local_begin(m_cursor);
      }
      return *this;
    }

    T& operator*() const { return *m_current; }
    bool operator== (cursor const& c) const
    {
      return m_current == c.m_current && m_cursor == c.m_cursor;
    }
    bool operator!= (cursor const& c) const { return !(*this == c); }
    
  private:
    segment_cursor<T> m_cursor;
    T*                m_current;
  };

  template <typename T>
  segment_cursor<T> segment(cursor<T> const& c) { return c.segment(); }
  template <typename T>
  T* local(cursor<T> const& c) { return c.local(); }

  template <typename T>
  T* local_begin(segment_cursor<T> const& c) { return c.begin(); }
  template <typename T>
  T* local_end(segment_cursor<T> const& c) { return c.end(); }
  template <typename T>
  cursor<T> make_segmented(segment_cursor<T> const& s, T* l) { return cursor<T>(s, l); }
}

namespace cool
{
  template <typename T>
  struct is_segmented<multipass::cursor<T> >
  {
    enum { value = true };
  };

  template <typename T>
  struct segment_cursor<multipass::cursor<T> >
  {
    typedef multipass::segment_cursor<T> type;
  };

  template <typename T>
  struct local_cursor<multipass::cursor<T> >
  {
    typedef T* type;
  };
}

// ----------------------------------------------------------------------- 

namespace c = cool;

namespace test
{
  void print(int i)
  {
    std::cout << i << " ";
  }

  void for_each()
  {
    int array[] = { 1, 2, 3, 4, 5 };

    std::cout << "=== for_each() ===\n";
    std::cout << "  normal/once:           ";
    c::for_each(c::begin(array), c::end(array), c::default_pm(), print);
    std::cout << "\n";

    std::cout << "  seg-complete:          ";
    c::for_each(multipass::cursor<int>(2, c::begin(array), c::end(array)),
                multipass::cursor<int>(c::end(array)), c::default_pm(), print);
    std::cout << "\n";

    std::cout << "  seg+offset:            ";
    c::for_each(multipass::cursor<int>(2, c::begin(array), c::end(array), c::begin(array) + 2),
                multipass::cursor<int>(c::end(array)), c::default_pm(), print);
    std::cout << "\n";

    std::cout << "  seg+offset+incomplete: ";
    c::for_each(multipass::cursor<int>(2, c::begin(array), c::end(array), c::begin(array) + 2),
                multipass::cursor<int>(c::end(array) - 2), c::default_pm(), print);
    std::cout << "\n";

    std::cout << "  seg+incomplete:        ";
    c::for_each(multipass::cursor<int>(2, c::begin(array), c::end(array)),
                multipass::cursor<int>(c::end(array) - 2), c::default_pm(), print);
    std::cout << "\n";

    std::cout << "  seg+negated:           ";
    c::for_each(multipass::cursor<int>(2, c::begin(array), c::end(array), c::begin(array)),
                multipass::cursor<int>(c::end(array)), std::negate<int>(), print);
    std::cout << "\n";
  }
}

// ----------------------------------------------------------------------- 

namespace test
{
  template <typename InCursor, typename ReadPM>
  void run_find(InCursor begin, InCursor end, ReadPM const& pm)
  {
    for (; end != (begin = cool::find(begin, end, pm, 2)); ++begin)
    {
      std::cout << "    ";
      cool::for_each(begin, end, pm, print);
      std::cout << "\n";
    }
  }

  void find()
  {
    namespace c = cool;
    int array[] = { 1, 2, 3, 4, 5 };

    std::cout << "=== find() ===\n";
    std::cout << "  normal\n";
    run_find(c::begin(array), c::end(array), c::default_pm());
    std::cout << "  segmented\n";
    run_find(multipass::cursor<int>(3, c::begin(array), c::end(array), c::begin(array)),
             multipass::cursor<int>(c::end(array)), c::default_pm());
    std::cout << "  segmented incomplete\n";
    run_find(multipass::cursor<int>(3, c::begin(array), c::end(array), c::begin(array)),
             multipass::cursor<int>(c::end(array) - 2), c::default_pm());
  }
}

// ----------------------------------------------------------------------- 

namespace test
{
  void copy()
  {
    namespace c = cool;
    int array[] = { 1, 2, 3, 4, 5 };
    int adest[6];
    std::vector<int> dest(1000);

    std::cout << "=== copy() ===\n";
    std::cout << "  normal/normal: ";
    c::copy(c::begin(array), c::end(array), c::default_pm(),
            std::ostream_iterator<int>(std::cout, " "), c::default_pm());
    std::cout << "\n";

    std::cout << "  seg/normal:    ";
    c::copy(multipass::cursor<int>(3, c::begin(array), c::end(array), c::begin(array)),
            multipass::cursor<int>(c::end(array)), c::default_pm(),
            std::ostream_iterator<int>(std::cout, " "), c::default_pm());
    std::cout << "\n";


    std::cout << "  seg/seg:       ";
    multipass::cursor<int> to =
      c::copy(multipass::cursor<int>(3, c::begin(array), c::end(array), c::begin(array)),
              multipass::cursor<int>(c::end(array)),
              std::bind1st(std::plus<int>(), 2),
              multipass::cursor<int>(10, c::begin(adest), c::end(adest), c::begin(adest)),
              c::default_pm());
    c::copy(multipass::cursor<int>(10, c::begin(adest), c::end(adest)), to, c::default_pm(),
            std::ostream_iterator<int>(std::cout, " "), c::default_pm());
    std::cout << "\n";
  }
}

// ----------------------------------------------------------------------- 

int main()
{
  test::for_each();
  test::find();
  test::copy();
}
