// -*-C++-*- find.hpp
// ----------------------------------------------------------------------- 
//  Copyright  2005 Dietmar Khl, All Rights Reserved                     
//                                                                         
//  Permission to use, copy, modify, distribute and sell this              
//  software for any purpose is hereby granted without fee, provided       
//  that the above copyright notice appears in all copies and that         
//  both that copyright notice and this permission notice appear in        
//  supporting documentation. Dietmar Khl makes no representations about  
//  the suitability of this software for any purpose. It is provided       
//  "as is" without express or implied warranty.                           
// ----------------------------------------------------------------------- 

// Author:  Dietmar Kuehl http://www.dietmar-kuehl.de 
// Title:   find() with cursors/property map and segment optimization

// ----------------------------------------------------------------------- 

#if !defined(COOL_FIND_HPP)
#define COOL_FIND_HPP 1

#include "enable_if.hpp"
#include "segments.hpp"
#include "default_pm.hpp"
#include "is_same.hpp"
#include <functional>
#include <algorithm>

namespace cool
{
  // ---------------------------------------------------------------------
  // default implementation of find_if()

  template <typename InCursor, typename ReadPM, typename UnaryPred>
  typename cool::enable_if<
      !cool::is_segmented<InCursor>::value
      && !is_same<ReadPM,default_pm>::value
    , InCursor
  >::type
  find_if(InCursor begin, InCursor end, ReadPM pm, UnaryPred pred)
  {
    while (begin != end && !pred(pm(*begin)))
      ++begin;
    return begin;
  };
  
  template <typename InCursor, typename ReadPM, typename UnaryPred>
  typename cool::enable_if<
      !cool::is_segmented<InCursor>::value
      && is_same<ReadPM,default_pm>::value
    , InCursor
  >::type
  find_if(InCursor begin, InCursor end, ReadPM, UnaryPred pred)
  {
      return std::find_if(begin, end, pred);
  };
  
  // ---------------------------------------------------------------------
  // specialization of find_if() for segmented sequences

  template <typename InCursor, typename ReadPM, typename UnaryPred>
  typename cool::enable_if<cool::is_segmented<InCursor>::value, InCursor>::type
  find_if(InCursor begin, InCursor end, ReadPM pm, UnaryPred pred)
  {
    typedef typename cool::segment_cursor<InCursor>::type segment_cursor;
    typename cool::local_cursor<InCursor>::type lbeg = local(begin);

    for (segment_cursor sbeg = segment(begin), send = segment(end);
         sbeg != send; lbeg = local_begin(++sbeg))
      if (local_end(sbeg) != (lbeg = cool::find_if(lbeg, local_end(sbeg), pm, pred)))
        return make_segmented(sbeg, lbeg);
    return make_segmented(segment(end), cool::find_if(local_begin(segment(end)),
                                                      local(end), pm, pred));
  }

  // ---------------------------------------------------------------------
  // find(): delegate to find_if()

  template <typename InCursor, typename ReadPM, typename T>
  InCursor
  find(InCursor begin, InCursor end, ReadPM pm, T const& v)
  {
    return cool::find_if(begin, end, pm, std::bind2nd(std::equal_to<T>(), v));
  }

  // ---------------------------------------------------------------------
}

#endif /* COOL_FIND_HPP */
