// -*-C++-*- copy.hpp
// ----------------------------------------------------------------------- 
//  Copyright  2005 Dietmar Khl, All Rights Reserved                     
//                                                                         
//  Permission to use, copy, modify, distribute and sell this              
//  software for any purpose is hereby granted without fee, provided       
//  that the above copyright notice appears in all copies and that         
//  both that copyright notice and this permission notice appear in        
//  supporting documentation. Dietmar Khl makes no representations about  
//  the suitability of this software for any purpose. It is provided       
//  "as is" without express or implied warranty.                           
// ----------------------------------------------------------------------- 

// Author:  Dietmar Kuehl http://www.dietmar-kuehl.de 
// Title:   copy() with cursors/property maps and segment optimization

// ----------------------------------------------------------------------- 

#if !defined(COOL_COPY_HPP)
#define COOL_COPY_HPP 1

#include "enable_if.hpp"
#include "segments.hpp"
#include "default_pm.hpp"
#include "is_same.hpp"
#include <iterator>
#include <iostream>
#include <algorithm>

namespace cool
{
  // ---------------------------------------------------------------------
  // default implementation of copy()
  
  template <typename InCursor, typename ReadPM,
            typename OutCursor, typename WritePM>
  typename cool::enable_if<
    !cool::is_segmented<InCursor>::value && !cool::is_segmented<OutCursor>::value
    && (!is_same<ReadPM,cool::default_pm>::value
        || !is_same<WritePM,cool::default_pm>::value),
    OutCursor
    >::type
  copy(InCursor begin, InCursor end, ReadPM read, OutCursor to, WritePM write)
  {
    for (; begin != end; ++begin, ++to)
      write(*to, read(*begin));
    return to;
  }

  template <typename InCursor, typename ReadPM,
            typename OutCursor, typename WritePM>
  typename cool::enable_if<
    !cool::is_segmented<InCursor>::value && !cool::is_segmented<OutCursor>::value
    && is_same<ReadPM,cool::default_pm>::value
        && is_same<WritePM,cool::default_pm>::value,
    OutCursor
    >::type
  copy(InCursor begin, InCursor end, ReadPM read, OutCursor to, WritePM write)
  {
      return std::copy(begin, end, to);
  }

  // ---------------------------------------------------------------------
  // specialization of copy() for the second sequence being segmented
  //-dk:TODO avoid use of this specialization if either the source or
  //         the local destination are not random access

  template <typename InCursor, typename ReadPM,
            typename OutCursor, typename WritePM>
  typename cool::enable_if<
    !cool::is_segmented<InCursor>::value && cool::is_segmented<OutCursor>::value,
    OutCursor
    >::type
  copy(InCursor begin, InCursor end, ReadPM read, OutCursor to, WritePM write)
  {
    typedef typename cool::local_cursor<OutCursor>::type local_cursor;
    typename cool::segment_cursor<OutCursor>::type sto = segment(to);
    local_cursor                                   lto = local(to);

    while (begin != end)
    {
      if (std::distance(begin, end) < std::distance(lto, local_end(sto)))
      {
        lto = cool::copy(begin, end, read, lto, write);
        break;
      }
      else
      {
        InCursor tmp(begin);
        std::advance(tmp, std::distance(lto, local_end(sto)));
        cool::copy(begin, tmp, read, lto, write);
        lto = local_begin(++sto);
        begin = tmp;
      }
    }

    return make_segmented(sto, lto);
  }

  // ---------------------------------------------------------------------
  // specialization of copy() for the first sequence being segmented

  template <typename InCursor, typename ReadPM,
            typename OutCursor, typename WritePM>
  typename cool::enable_if<cool::is_segmented<InCursor>::value, OutCursor>::type
  copy(InCursor begin, InCursor end, ReadPM read, OutCursor to, WritePM write)
  {
    typedef typename cool::segment_cursor<InCursor>::type segment_cursor;
    typename cool::local_cursor<InCursor>::type lbeg = local(begin);

    for (segment_cursor sbeg = segment(begin), send = segment(end);
         sbeg != send; lbeg = local_begin(++sbeg))
      to = cool::copy(lbeg, local_end(sbeg), read, to, write);
    return cool::copy(local_begin(segment(end)), local(end), read, to, write);
  }

  // ---------------------------------------------------------------------
}

#endif /* COOL_COPY_HPP */
