//-----------------------------------------------------------------------------
// boost mpl/runtime/arithmetic/pow.hpp.hpp header file
// See http://www.boost.org for updates, documentation, and revision history.
//-----------------------------------------------------------------------------
//
// Copyright (c) 2002
// Terje Sletteb
//
// Permission to use, copy, modify, distribute and sell this software
// and its documentation for any purpose is hereby granted without fee,
// provided that the above copyright notice appears in all copies and
// that both the copyright notice and this permission notice appear in
// supporting documentation. No representations are made about the
// suitability of this software for any purpose. It is provided "as is"
// without express or implied warranty.

#ifndef BOOST_MPL_RUNTIME_ARITHMETIC_POW_HPP_INCLUDED
#define BOOST_MPL_RUNTIME_ARITHMETIC_POW_HPP_INCLUDED

#include <cmath>
#include "pow.hpp"
#include "boost/mpl/if.hpp"
#include "boost/mpl/identity.hpp"

namespace boost {
namespace mpl {
namespace runtime {

namespace aux {

///////////////////////////////////////////////////////////////////////////////
// runtime_pow
//
// Helper class for runtime_pow_impl
//
// pow(run-time,run-time)
//
// May specialise for integral exponent, as well.
// Requires partial specialisation or simulation of it, so it's left out.
///////////////////////////////////////////////////////////////////////////////

template<class Base,class Exp>
struct runtime_pow
{
  typedef identity<Base> type;

  static Base result(Base base,Exp exp)
  {
    return std::pow(base,exp);
  }
};

///////////////////////////////////////////////////////////////////////////////
// runtime_pow_base
//
// Helper class for pow_exp_constant
///////////////////////////////////////////////////////////////////////////////

template<typename Base,typename Exp>
struct runtime_pow_base
{
  template<typename ExpValue>
  struct apply
  {
    typedef apply type;

    static Base result(Base base) { return 1; }
  };
};

///////////////////////////////////////////////////////////////////////////////
// runtime_pow_recursive
//
// Helper class for pow_exp_constant
///////////////////////////////////////////////////////////////////////////////

template<typename Base,typename Exp>
struct runtime_pow_recursive
{
  template<typename ExpValue>
  struct apply
  {
    typedef apply type;

    static Base result(Base base)
    {
      return base * apply1<
        pow_exp_constant<Base,Exp>,
        prior<ExpValue>::type
      >::type::result(base);
    }
  };
};

///////////////////////////////////////////////////////////////////////////////
// pow_exp_constant
//
// Helper class for runtime_pow_impl
//
// pow(run-time,compile-time)
///////////////////////////////////////////////////////////////////////////////

template<typename Base,typename Exp>
struct pow_exp_constant
{
  typedef identity<Base> type;

  template<typename ExpValue>
  struct apply
  {
    typedef apply type;

    static Base result(Base base)
    {
      typedef typename if_<
        equal_to<ExpValue,value<ExpValue,0>::type>,
        runtime_pow_base<Base,Exp>,
        runtime_pow_recursive<Base,Exp>
      >::type evaluator;

      return apply1<evaluator,ExpValue>::type::result(base);
    }
  };

  static Base result(Base base,Exp exp)
  {
    return apply1<pow_exp_constant<Base,Exp>,Exp>::type::result(base);
  }
};

///////////////////////////////////////////////////////////////////////////////
// runtime_pow_impl
///////////////////////////////////////////////////////////////////////////////

template<class Base,class Exp>
struct runtime_pow_impl
{
  typedef typename if_<
    has_ct_tag<Exp>,               // Exp==constant?
    typename if_<
      has_ct_tag<Base>,            // Base==constant?
      ::boost::mpl::pow<Base,Exp>, // pow(constant,constant)
      pow_exp_constant<Base,Exp>   // pow(runtime,constant)
    >::type,
    runtime_pow<Base,Exp>          // pow(runtime,runtime)
  >::type type;
};

} // namespace aux

///////////////////////////////////////////////////////////////////////////////
// pow
///////////////////////////////////////////////////////////////////////////////

template<class Base,class Exp>
inline typename aux::runtime_pow_impl<Base,Exp>::type::type::type pow(Base base,Exp exp)
{
  return aux::runtime_pow_impl<Base,Exp>::type::result(base,exp);
}

} // namespace runtime
} // namespace mpl
} // namespace boost

#endif // BOOST_MPL_RUNTIME_ARITHMETIC_POW_HPP_INCLUDED
