#ifndef _SERIAL_H
#define _SERIAL_H


//#include <boost/bind.hpp>
#include <boost/asio.hpp>
#include <boost/asio/serial_port.hpp>
#include <boost/thread.hpp>
#include <boost/signals2/signal.hpp>
#include <boost/scoped_ptr.hpp>
#include <boost/enable_shared_from_this.hpp>


#define READ_TIMEOUT  1000  //msec

#define wxCOM1 "com1"
#define wxCOM2 "com2"
#define wxCOM3 "com3"
#define wxCOM4 "com4"
#define wxCOM5 "com5"
#define wxCOM6 "com6"
#define wxCOM7 "com7"
#define wxCOM8 "com8"
#define wxCOM9 "com9"
#define wxCOM10 "\\\\.\\com10"
#define wxCOM11 "\\\\.\\com11"
#define wxCOM12 "\\\\.\\com12"
#define wxCOM13 "\\\\.\\com13"
#define wxCOM14 "\\\\.\\com14"
#define wxCOM15 "\\\\.\\com15"
#define wxCOM16 "\\\\.\\com16"


/*!
  \enum SerialLineState

  Defines the different modem control lines.
  \see wxSerialLineState
*/
enum SerialLineState
{
    /*! Data Carrier Detect (read only) */
    SERIAL_LINESTATE_DCD = 0x040,
    /*! Clear To Send (read only) */
    SERIAL_LINESTATE_CTS = 0x020,
    /*! Data Set Ready (read only) */
    SERIAL_LINESTATE_DSR = 0x100,
    /*! Data Terminal Ready (write only) */
    SERIAL_LINESTATE_DTR = 0x002,
    /*! Ring Detect (read only) */
    SERIAL_LINESTATE_RING = 0x080,
    /*! Request To Send (write only) */
    SERIAL_LINESTATE_RTS = 0x004,
    /*! no active line state, use this for clear */
    SERIAL_LINESTATE_NULL = 0x000
};

/*!
  \enum DevParity

  Defines the different modes of parity checking. Under
  Linux, the struct termios will be set to provide the wanted
  behaviour.
*/
enum DevParity
{
  /*! no parity check */
  devPARITY_NONE,
  /*! odd parity check */
  devPARITY_ODD,
  /*! even parity check */
  devPARITY_EVEN
};

/*!
  \enum DevFlowControl

  Defines the flow control to be used in RS232 communication.
*/
enum DevFlowControl
{
  /*! no flow control */
  FLOW_NONE,
  /*! hardware flow control */
  FLOW_RTS_CTS,
  /*! software flow control */
  FLOW_XON_XOFF,
  /*! hardware and software flow control */
  FLOW_RTS_XONOFF
};

/*!
  \struct SerialDCS

  The device control struct for the serial communication.
*/
struct SerialDCS {
  /*! the baudrate */
  INT            baud;
  /*! the parity */
  DevParity      parity;
  /*! count of stopbits */
  UCHAR          stopbits;
  /*! flow control */
  DevFlowControl flowcontrol;

  // Constructor
  inline SerialDCS() {
    baud        = 9600;
    parity      = devPARITY_NONE;
    stopbits    = 1;
    flowcontrol = FLOW_NONE;
  };
  // Destructor to avoid memory leak
  ~SerialDCS() {};
};


/*!
  \struct SerialLines

  This struct defines the read-only lines of RS232.
*/
struct SerialLines {
  BOOL Line_DCD;
  BOOL Line_CTS;
  BOOL Line_DSR;
  BOOL Line_RING;
  
  BOOL operator==(SerialLines sl)
  {
    return ( (Line_DCD == sl.Line_DCD) && 
             (Line_CTS == sl.Line_CTS) && 
             (Line_DSR == sl.Line_DSR) && 
             (Line_RING == sl.Line_RING) );
  };
  BOOL operator!=(SerialLines sl)
  {
    return ( (Line_DCD != sl.Line_DCD) || 
             (Line_CTS != sl.Line_CTS) || 
             (Line_DSR != sl.Line_DSR) || 
             (Line_RING != sl.Line_RING) );
  };
};


typedef boost::signals2::connection               _signal_conn;     //Define a signal connection

typedef boost::signals2::signal<void ()>          _signal_void;     //Define a signal type for OnData event
typedef boost::signals2::signal<void (INT, BOOL)> _signal_int_bool; //Define a signal type for OnLineState

typedef _signal_void::slot_type         _slot_type_void;      //Define a slot type for connecting to the signal void
typedef _signal_int_bool::slot_type     _slot_type_int_bool;  //Define a slot type for connecting to the signal (int, bool)

//typedef std::vector<CHAR>               TBUFFER;
typedef std::string                     TBUFFER;
  

/*!
  \class SerialPort
  This Class wrap the wxSerialPort from ctb library.
 */
class SerialPort : public boost::enable_shared_from_this<SerialPort>
{

public:
  /*************************** PROPERTIES *****************************/

  /*!
    \brief Return the selected COM port number.
    \return COM port selected.
   */
  CHAR GetComPortNo(void);
  /*!
    \brief Return the selected COM port name.
    \return COM port selected.
   */
  std::string GetComPortStr(void);


  /*!
    \brief Set the COM port device.
    \param ComNo: New COM port number
   */
  void SetComPort(CHAR ComNo);
  /*!
    \brief Set the COM port device.
    \param DevName: New COM port name (eg. "\\\\.\\COM1")
   */
  void SetComPort(std::string DevName);

  /*!
    \brief Get the size of RX buffer.
    \return The current RX buffer size.
   */
  INT GetBufferSize(void);
  /*!
    \brief Set the size of RX buffer.
    \param value: The new RX buffer size.
   */
  void SetBufferSize(INT value);


  /*!
    \brief Get the Device Control Structure.
    \param DCS: A pointer to the returned struct.
   */
  void GetSerialDCS(SerialDCS *DCS);
  /*!
    \brief Set a new Device Control Structure.
    To take the edit be applied, close and re-open the device.
    \param DCS: A pointer to the new DCS
   */
  void SetSerialDCS(SerialDCS *DCS);


  /*!
    \brief Set the new state of the RS232 DTR Pin.
    \param value: New state of DTR: False -> off, True -> On.
   */
  void SetDTR(BOOL value);

  /*!
    \brief Set the new state of the RS232 RTS Pin.
    \param value: New state of RTS: False -> off, True -> On.
   */
  void SetRTS(BOOL value);

  /*!
    \brief Get the state of the RS232 lines (CDC,CTS,DSR,RING).
    \return A SerialLines struct which contains the lines state.
   */
  SerialLines GetLineState(void);


  /*************************** METHODS *****************************/

  /*!
    \brief Constructors.
   */
  SerialPort(boost::asio::io_service &_io);
  SerialPort(boost::asio::io_service &_io, INT BufferSize);

  /*!
    \brief Destructor.
   */
  ~SerialPort();

  /*!
    \brief Write the buffer on the Serial device.
    \param buffer: Pointer to the buffer to be wrote.
    \return True on success, False otherwise.
   */
  BOOL Write(TBUFFER buffer);

  /*!
    \brief Read the incoming buffer.
    \param Pointer to a vector where the buffer must be copied into.
   */
  void GetData(TBUFFER &buffer);
  //TBUFFER GetData(void);

  /*!
    \brief Open the device.
    \return True on success, False otherwise.
   */
  BOOL Open(void);  
  
  /*!
    \brief Open the device.
    \param ComNo: Optional new device number.
    \return True on success, False otherwise.
   */
  BOOL Open(CHAR ComNo);

  /*!
    \brief Open the device.
    \param DevName: Optional DevName new device name (eg. "\\\\.\\COM1").
    \return True on success, False otherwise.
   */
  BOOL Open(std::string DevName);

  /*!
    \brief Close the device.
    \return True on success, False otherwise.
   */
  BOOL Close(void);

  /*!
    \brief Connection point to a call-back that receive OnData event.
           This event is raised when some data is readed from the device
           and is ready to be catched by GetData member.
    \param A reference to a connection Object.
    \return The new connection object created.
   */
  _signal_conn OnData(const _slot_type_void& s);

  /*!
    \brief Connection point to a call-back that receive OnLineState event.
           This event is raised when one of the RS232 lines changes its state.
    \param A reference to a connection Object.
   */
  _signal_conn OnLineState(const _slot_type_int_bool& s);

private:

  /*!
    \brief
  */
  //void init_dcs();

  void read_complete(const boost::system::error_code& error, size_t bytes_transferred);
  void do_write(TBUFFER &buffer);
  void Post_OnData();
  void timer_exipred(const boost::system::error_code& error);

public:

  static const char     CTRL_Z  = 26;


  /*!
  \brief .  
 */
  bool                  UseEosString;

  /*!
  \brief .  
 */
  std::string           EosString;

  /*!
  \brief .  
 */
  UINT                  ReadTimeOut;

private:
  //wxSerialPort          m_serial_port;
  //wxSerialPort_DCS      wx_dcs;

  SerialDCS                     _dcs;

  boost::asio::io_service&      _io_service;  // the main IO service that runs this connection
  boost::asio::serial_port      _serialPort;  // the serial port this instance is connected to
  boost::asio::deadline_timer   _rx_timer;

  size_t                        m_buffer_size;
  std::string                   m_ComPort;
  TBUFFER                       _read_buffer;
  PCHAR                         _buf;

  //volatile bool                    _stoprequested;
  //boost::mutex                     _mutex;
  //boost::asio::io_service::work    _io_work;
  //boost::scoped_ptr<boost::thread> _thread;

  _signal_void        signal_OnData;      //Raised when data is readed
  _signal_int_bool    signal_OnLineState; //Raised when a line has changed its state

  // start reading from serial port
  void reading(void);
};

#endif  //_SERIAL_H
