/////////////////////////////////////////////////////////////////////////////
// Name:        Serial.cpp
// Purpose:
// Author:      Daniele Barzotti
// Id:          $Id: Serial.cpp,v 1.0.0.0 2009/02/-- jb Exp $
// Copyright:   (c) 2009 Eurocom Telecomunicazioni
// Licence:
/////////////////////////////////////////////////////////////////////////////

#include "Serial.hpp"
#include <cstring>
#include <vector>
#include <algorithm>

using namespace boost::asio;

// Map to associate the strings with the enum values
typedef std::vector<std::string> str_vector;

static str_vector s_DevNames;

static void InitDeviceNames(void)
{
  static bool init = false;

  if (init) return;
  init = true;

  s_DevNames.reserve(20);
  s_DevNames.push_back("0");  //Dummy value: not valid!
  s_DevNames.push_back(wxCOM1);
  s_DevNames.push_back(wxCOM2);
  s_DevNames.push_back(wxCOM3);
  s_DevNames.push_back(wxCOM4);
  s_DevNames.push_back(wxCOM5);
  s_DevNames.push_back(wxCOM6);
  s_DevNames.push_back(wxCOM7);
  s_DevNames.push_back(wxCOM8);
  s_DevNames.push_back(wxCOM9);
  s_DevNames.push_back(wxCOM10);
  s_DevNames.push_back(wxCOM11);
  s_DevNames.push_back(wxCOM12);
  s_DevNames.push_back(wxCOM13);
  s_DevNames.push_back(wxCOM14);
  s_DevNames.push_back(wxCOM15);
  s_DevNames.push_back(wxCOM16);

};


//--------------------------------------------------------------------------------------


void SerialPort::Post_OnData()
{
  // Raise the event
  signal_OnData();
};

//--------------------------------------------------------------------------------------

void SerialPort::timer_exipred(const boost::system::error_code& error)
{
  if (!error)
  {
    // Timer expired-> read completed, so process the data
    _io_service.post( boost::bind(&SerialPort::Post_OnData, this) );
  }
}

//--------------------------------------------------------------------------------------

void SerialPort::read_complete(const boost::system::error_code& error, size_t bytes_transferred)
{
  // the asynchronous read operation has now completed or failed and returned an error
  if (!error)
  { 
    // saving data to vector        
    _read_buffer.append(_buf, bytes_transferred);

    //Start a new timer or renew it
    _rx_timer.expires_from_now( boost::posix_time::milliseconds(ReadTimeOut) );
    // We managed to cancel the timer. Start new asynchronous wait.
    _rx_timer.async_wait( boost::bind(&SerialPort::timer_exipred, this, _1) );

    // start waiting for another asynchronous read again
    reading(); 
  }
  //else
  //  signal_OnError(error);
};

//--------------------------------------------------------------------------------------

void SerialPort::reading()
{
  // Start an asynchronous read and call read_complete when it completes or fails
  _serialPort.async_read_some(boost::asio::buffer(_buf, m_buffer_size),
                              boost::bind(&SerialPort::read_complete,
                              this,
                              boost::asio::placeholders::error,
                              boost::asio::placeholders::bytes_transferred));
};

//--------------------------------------------------------------------------------------

void SerialPort::SetDTR(BOOL value)
{
  //// Retreive the current lines state
  //int LineState = m_serial_port.GetLineState();
  //if (value)
  //  LineState |= SERIAL_LINESTATE_DTR;
  //else
  //  LineState &= (!SERIAL_LINESTATE_DTR);
  //// set the new states
  //m_serial_port.ChangeLineState(static_cast<wxSerialLineState>(LineState));
};

//--------------------------------------------------------------------------------------

void SerialPort::SetRTS(BOOL value)
{
  //// Retreive the current lines state
  //int LineState = m_serial_port.GetLineState();
  //if (value)
  //  LineState |= SERIAL_LINESTATE_RTS;
  //else
  //  LineState &= (!SERIAL_LINESTATE_RTS);
  //// set the new states
  //m_serial_port.ChangeLineState(static_cast<wxSerialLineState>(LineState));
};

//--------------------------------------------------------------------------------------

void SerialPort::SetBufferSize(INT value)
{
  m_buffer_size = value;

  if (_buf) delete[] _buf;
  _buf = new char[m_buffer_size];
  memset(_buf, 0, m_buffer_size);
};

//--------------------------------------------------------------------------------------

INT SerialPort::GetBufferSize(){
  return m_buffer_size;
};

//--------------------------------------------------------------------------------------

SerialLines SerialPort::GetLineState(void)
{
  SerialLines sl;
  //int LineState = m_serial_port.GetLineState();

  //sl.Line_DCD = (LineState & SERIAL_LINESTATE_DCD);
  //sl.Line_CTS = (LineState & SERIAL_LINESTATE_CTS);
  //sl.Line_DSR = (LineState & SERIAL_LINESTATE_DSR);
  //sl.Line_RING = (LineState & SERIAL_LINESTATE_RING);

  return sl;
};

//--------------------------------------------------------------------------------------

SerialPort::SerialPort(boost::asio::io_service &_io, INT BufferSize) : 
                       m_buffer_size(BufferSize), 
                       _io_service(_io),
                       _serialPort(_io_service), 
                       _rx_timer(_io_service),
                       ReadTimeOut(READ_TIMEOUT)
{
  //init_dcs();
  EosString = '\n';
  _buf = 0;
};

//--------------------------------------------------------------------------------------

SerialPort::SerialPort(boost::asio::io_service &_io) : 
                       m_buffer_size(100), 
                       _io_service(_io),
                       _serialPort(_io_service), 
                       _rx_timer(_io_service),
                       ReadTimeOut(READ_TIMEOUT)
{
  //init_dcs();
  EosString = '\n';
  _buf = 0;
};

//--------------------------------------------------------------------------------------

SerialPort::~SerialPort()
{
  SerialPort::Close();
};

//--------------------------------------------------------------------------------------

CHAR SerialPort::GetComPortNo(void)
{  
  str_vector::const_iterator it;
  it = std::find(s_DevNames.begin(), s_DevNames.end(), m_ComPort); 
  return (it - s_DevNames.begin());
};

//--------------------------------------------------------------------------------------

std::string SerialPort::GetComPortStr(void)
{
  return m_ComPort;
};
//--------------------------------------------------------------------------------------

void SerialPort::SetComPort(CHAR ComNo)
{
  if (ComNo > 0) m_ComPort = s_DevNames[ComNo];
};

//--------------------------------------------------------------------------------------

void SerialPort::SetComPort(std::string DevName)
{
  m_ComPort = DevName;
};
//--------------------------------------------------------------------------------------


void SerialPort::GetSerialDCS(SerialDCS *DCS)
{
  memcpy(DCS, &_dcs, sizeof(SerialDCS));
};

//--------------------------------------------------------------------------------------

void SerialPort::SetSerialDCS(SerialDCS *DCS)
{
  memcpy(&_dcs, DCS, sizeof(SerialDCS));

  try {
    if (_serialPort.is_open())
    {
      serial_port_base::baud_rate baud_option(_dcs.baud);
      _serialPort.set_option( baud_option );

      serial_port_base::baud_rate flow_option(_dcs.flowcontrol);
      _serialPort.set_option( flow_option );

      serial_port_base::baud_rate parity_option(_dcs.parity);
      _serialPort.set_option( parity_option );

      serial_port_base::baud_rate stopbits_option(_dcs.stopbits);
      _serialPort.set_option( stopbits_option );
    }
  }
  catch (...) {}
};

//--------------------------------------------------------------------------------------

BOOL SerialPort::Write(TBUFFER buffer)
{
  size_t byte_written = boost::asio::write(_serialPort, boost::asio::buffer(buffer.c_str(), buffer.size()));
  return (byte_written == buffer.size());
};

//--------------------------------------------------------------------------------------

//void SerialPort::do_write(TBUFFER &buffer)
//{
//  size_t byte_written = boost::asio::write(_serialPort, boost::asio::buffer(buffer.c_str(), buffer.size()));
//  //(byte_written == buffer.size())
//};

//--------------------------------------------------------------------------------------

void SerialPort::GetData(TBUFFER& buffer)
{
  // Lock the read buffer vector  
  //buffer.assign(_read_buffer.begin(), _read_buffer.end());
  //Clear the read buffer
  _read_buffer.clear();  
};

//--------------------------------------------------------------------------------------

BOOL SerialPort::Open(void)
{
  boost::system::error_code ec;
  ec = _serialPort.open(m_ComPort, ec);
  if (!_serialPort.is_open()) return false;

  SetSerialDCS(&_dcs);  
  SetBufferSize(m_buffer_size);

  this->reading();
  return true;
};

//--------------------------------------------------------------------------------------

BOOL SerialPort::Open(std::string DevName)
{  
  if (DevName.length() > 0) SetComPort(DevName);
  return Open();
};

//--------------------------------------------------------------------------------------

BOOL SerialPort::Open(CHAR ComNo)
{
  if (ComNo > 0) SetComPort(ComNo);
  return Open();
};

//--------------------------------------------------------------------------------------

BOOL SerialPort::Close(void)
{
  if (!_serialPort.is_open())
  {
    // If the port is open, then the reading thread is
    //io_service_.post(boost::bind(&SerialPort::do_close, this, boost::system::error_code()));
   _serialPort.close();
  }
  return true;
};


//--------------------------------------------------------------------------------------

_signal_conn SerialPort::OnData(const _slot_type_void& s)
{
  return signal_OnData.connect(s);
};

//--------------------------------------------------------------------------------------

_signal_conn SerialPort::OnLineState(const _slot_type_int_bool& s)
{
  return signal_OnLineState.connect(s);
};

//--------------------------------------------------------------------------------------
