#ifndef _ATDISPATCHER_H
#define _ATDISPATCHER_H


#ifdef _USE_SERIAL_OK
  #include "Serial_OK.hpp"
#else
  #include "Serial.hpp"
#endif

#include <boost/lambda/lambda.hpp>
#include <boost/bind.hpp>
//#include <boost/regex.hpp>
#include <boost/thread.hpp>
#include <boost/thread/condition_variable.hpp>
#include <boost/thread/xtime.hpp>


  #define COMMAND_TIMEOUT  5  // 5 sec.

  typedef std::vector<std::string>  PARSED_BUFFER;

  /*!
    \class ATDispatcher.
    This class is a dispatcher for AT commands, it send and receive commands to/from RS232
   */
  class ATDispatcher
  {
    public:

      
      enum EDispatcherRetVal
      {
        RET_OK,
        RET_TIMEOUT,
        RET_ERROR
      };


      /*!
      \brief Destructor.    
     */
      ~ATDispatcher(void);
      
     /*!
      \brief Constructor.
      \param device A pointer to a SerialPort class.
     */
      ATDispatcher(SerialPort& device);


     ///*!
     // \brief ParseBuffer property. It define if the buffer has to be parsed.
     //*/
     // bool GetParseBuffer();
     // void SetParseBuffer(bool value);


      /*!
      \brief                Sends out the AT command to the serial port.
                            This function is a blocking function and do not returns until
                            the modem is answered or the timeout is raised.
      \param cmd            The AT command to send.
      \param waitOK         Indicates if the dispatcher has to expect an 'OK' in the answer.
      \param parse_answer   Indicates if the dispatcher has to decode (parse) the answer.
      \param append_CR      Indicates if the dispatcher has to append a \r\n to the command sent.
      \param forward_answer Indicates if the dispatcher has to foreword the answer trough OnData event.
      \param cmd_timeout    Indicates the time to wait for the modem answer.
      \return               \see EDispatcherRetVal.
     */
      EDispatcherRetVal SendCommand(std::string& cmd, 
                                    bool waitOK = true, 
                                    bool parse_answer = true,
                                    bool append_CR = true,
                                    bool forward_answer = false,
                                    INT  cmd_timeout = COMMAND_TIMEOUT);

      EDispatcherRetVal SendCommand(const char* cmd, 
                                    bool waitOK = true, 
                                    bool parse_answer = true,
                                    bool append_CR = true,
                                    bool forward_answer = false,
                                    INT  cmd_timeout = COMMAND_TIMEOUT);

      //EDispatcherRetVal SendCommand(const char* cmd, 
      //                              bool waitOK, 
      //                              bool parse_answer,
      //                              bool append_CR,
      //                              INT  cmd_timeout);

      /*!
      \brief Get the raw answer (not parsed) from the modem.
      \param answer A reference to a buffer that has to receive the answer.
     */
      void GetRawData(std::string& buffer);

      /*!
      \brief Get the answer splitted into a vector.
      \param answer A reference to a string vector that has to receive the answer.
     */
      void GetParsedData(std::vector<std::string>& buffer);

      /*!
        \brief Connection point to a call-back that receive OnData event.
               This event is raised when some data is readed from the device
               and is ready to be catched by GetData member.
        \param A reference to a connection Object.
        \return The new connection object created.
       */
      _signal_conn OnData(const _slot_type_void& s);

    private:
      /*!
      \brief Connection point to a call-back that receive OnDataReveived event from Serial class.
      */
      void OnDataReceived();
    
      /*!
      \brief Initialize variables.
      */
      void ResetVars();


    public:
      std::string               StartString;
      std::string               EndString;
      bool                      ParseBuffer;

    private:
      std::string               _buffer;
      SerialPort&               _device;
      boost::condition_variable _cond;
      boost::mutex              _mut;
      //boost::mutex              _cond_mutex;
      PARSED_BUFFER             _parsed_buffer;
      bool                      _data_ready;
      bool                      _waitOK;
      bool                      _waiting_answer;
      bool                      _forward_answer;
      bool                      _error;
      
      boost::scoped_ptr<boost::thread>  _read_thread_ptr;

      _signal_void              signal_OnData;      //Raised when data is readed

  };


#endif //_ATDISPATCHER_H