#ifndef NON_FINITE_NUM_FACETS_HPP
#define NON_FINITE_NUM_FACETS_HPP

// non_finite_num_facets.hpp

// Implementation of the class templates num_put_extended and num_get_extended

// Copyright (c) 2006 Johan Rde
// Distributed under the Boost Software License, Version 1.0. 
// (See http://www.boost.org/LICENSE_1_0.txt)

#include <ios>
#include <iterator>
#include <limits>
#include <locale>

// num_put_extended

template<
	class CharType, 
	class OutputIterator = std::ostreambuf_iterator<CharType> 
>
class num_put_extended : public std::num_put<CharType, OutputIterator> {
protected:
	virtual iter_type do_put(
		iter_type it, std::ios_base& iosb, CharType fill, float val) const
	{
		return do_put_impl(it, iosb, fill, val);
	}

	virtual iter_type do_put(
		iter_type it, std::ios_base& iosb, CharType fill, double val) const
	{
		return do_put_impl(it, iosb, fill, val);
	}

	virtual iter_type do_put(
		iter_type it, std::ios_base& iosb, CharType fill, long double val) const
	{
		return do_put_impl(it, iosb, fill, val);
	}

private:
	template<class ValType> iter_type do_put_impl(
		iter_type it, std::ios_base& iosb, CharType fill, ValType val) const
	{
		// On some platforms x == x is false for x = quite_NaN or signaling_NaN
		if(!(val == val))
			return put_nan(it);

		if(std::numeric_limits<ValType>::has_infinity) {
			if(val == std::numeric_limits<ValType>::infinity())
				return put_inf(it);
			else if(val == -std::numeric_limits<ValType>::infinity()) {
				*it = '-';
				return put_inf(it);
			}
		}

		if(std::numeric_limits<ValType>::has_quiet_NaN) {
			if(val == std::numeric_limits<ValType>::quiet_NaN())
				return put_nan(it);
		}

		if(std::numeric_limits<ValType>::has_signaling_NaN) {
			if(val == std::numeric_limits<ValType>::signaling_NaN())
				return put_nan(it);
		}

		return std::num_put<CharType, OutputIterator>::do_put(it, iosb, fill, val);
	}

	iter_type put_inf(iter_type it) const
	{
		*it = 'i';
		*it = 'n';
		*it = 'f';
		return it;
	}

	iter_type put_nan(iter_type it) const
	{
		*it = 'n';
		*it = 'a';
		*it = 'n';
		return it;
	}
};

// num_get_extended

template<
	class CharType, 
	class InputIterator = std::istreambuf_iterator<CharType> 
>
class num_get_extended : public std::num_get<CharType, InputIterator> {
protected:
	virtual iter_type do_get(
		iter_type it, iter_type end, std::ios_base& iosb,
		std::ios_base::iostate& state, float& val) const
	{
		return do_get_impl(it, end, iosb, state, val);
	}

	virtual iter_type do_get(
		iter_type it, iter_type end, std::ios_base& iosb,
		std::ios_base::iostate& state, double& val) const
	{
		return do_get_impl(it, end, iosb, state, val);
	}

	virtual iter_type do_get(
		iter_type it, iter_type end, std::ios_base& iosb,
	    std::ios_base::iostate& state, long double& val) const
	{
		return do_get_impl(it, end, iosb, state, val);
	}
	
private:
	template<class ValType> iter_type do_get_impl(
		iter_type it, iter_type end, std::ios_base& iosb,
		std::ios_base::iostate& state, ValType& val) const
	{
		if(it == end) {
			state |= std::ios_base::failbit | std::ios_base::eofbit;
			return it;
		}

		char_type c = *it;

		if(c == 'n') 
			it = get_nan_tail(it, end, iosb, state, val);

		else {
			bool negative = (c == '-');
			if(negative || c == '+') {
				++it;
				if(it == end) {
					state |= std::ios_base::failbit | std::ios_base::eofbit;
					return it;
				}
				c = *it;
			}

			if(c == 'i')
				it = get_inf_tail(it, end, iosb, state, val);
			else
				it = std::num_get<CharType, InputIterator>::do_get(it, end, iosb, state, val);

			if(negative && !(state & std::ios_base::failbit))
				val = -val;
		}

		if(it == end)
			state |= std::ios_base::eofbit;

		return it;
	}

	template<class ValType> iter_type get_inf_tail(
		iter_type it, iter_type end, std::ios_base& iosb,
		std::ios_base::iostate& state, ValType& val) const
	{
		++it;
		if(std::numeric_limits<ValType>::has_infinity) {
			if(it != end && *it == 'n') {
				++it;
				if(it != end && *it == 'f') {
					++it;
					val = std::numeric_limits<ValType>::infinity();
					return it;
				}
			}
		}			
		state |= std::ios_base::failbit;
		return it;
	}

	template<class ValType>
	iter_type get_nan_tail(
		iter_type it, iter_type end, std::ios_base& iosb,
		std::ios_base::iostate& state, ValType& val) const
	{
		++it;
		if(std::numeric_limits<ValType>::has_quiet_NaN) {
			if(it != end && *it == 'a') {
				++it;
				if(it != end && *it == 'n') {
					++it;
					val = std::numeric_limits<ValType>::quiet_NaN();
					return it;
				}
			}
		}			
		state |= std::ios_base::failbit;
		return it;
	}
};

#endif

